#!/bin/bash

# 当任何命令失败时，立即退出脚本
set -e

# --- 配置 ---
# 你的服务和对应的健康检查URL
# 格式: "服务名:宿主机端口"
SERVICES=(
  "black-s1:8089"
  "black-s2:8099"
)
HEALTH_CHECK_PATH="/" # Next.js应用的根路径通常返回200 OK
MAX_WAIT_SECONDS=60 # 等待服务启动的最长时间（秒）
PROJECT_NAME="dy-black"

# --- 脚本函数 ---

# 函数：等待服务启动并健康
# 参数1: 服务名 (e.g., black-s1)
# 参数2: 宿主机端口 (e.g., 8089)
wait_for_service() {
  local service_name=$1
  local host_port=$2
  local start_time=$(date +%s)

  echo ">>> 正在等待服务 '$service_name' 在端口 '$host_port' 上启动..."

  while true; do
    # 使用 curl 检查服务是否返回 200 OK
    if curl -s -f "http://127.0.0.1:${host_port}${HEALTH_CHECK_PATH}" > /dev/null; then
      echo "✅ 服务 '$service_name' 已成功启动！"
      return 0
    fi

    # 检查是否超时
    local current_time=$(date +%s)
    local elapsed_time=$((current_time - start_time))
    if [ $elapsed_time -ge $MAX_WAIT_SECONDS ]; then
      echo "❌ 错误: 等待服务 '$service_name' 超时 (超过 ${MAX_WAIT_SECONDS} 秒)。"
      echo "请检查容器日志: docker logs ${service_name}"
      exit 1
    fi

    # 等待1秒后重试
    sleep 1
    echo -n "." # 打印进度点
  done
}

# --- 主逻辑 ---

echo "🚀 开始滚动重启..."

# 循环更新每个服务
for service_info in "${SERVICES[@]}"; do
  # 从 "服务名:端口" 中分离出服务名和端口
  IFS=':' read -r service_name port <<< "$service_info"

  echo ""
  echo "=========================================================="
  echo "  正在重启容器: ${service_name}"
  echo "=========================================================="
  
  # 使用 docker compose 更新并重建单个服务，-d 表示后台运行，--no-deps 表示不启动依赖
  docker compose -p ${PROJECT_NAME} restart ${service_name}

  # 等待该服务健康
  wait_for_service ${service_name} ${port}
done

echo ""
echo "🎉 全部服务重启完成！"