import bundleAnalyzer from '@next/bundle-analyzer';

const withBundleAnalyzer = bundleAnalyzer({
    enabled: process.env.ANALYZE === 'true',
});

/** @type {import('next').NextConfig} */
const nextConfig = {
    assetPrefix: process.env.NODE_ENV === 'development' ? '' : 'https://ggjs.bdrcp.cn/black/v1_1',
    async headers() {
        return [
            {
                // 匹配所有 _next/static 目录下的静态资源 (JS, CSS, fonts, etc.)
                // :path* 是一个通配符，匹配该路径下的所有内容
                source: '/_next/static/:path*',
                headers: [
                    {
                        key: 'Cache-Control',
                        // 关键：public, max-age=一年, immutable
                        value: 'public, max-age=31536000, immutable',
                    },
                ],
            },
            {
                // 匹配页面路由 (例如: /, /about, /products/123)
                // 使用一个更通用的正则表达式来匹配非 API 和非静态资源的路径
                source: '/((?!api|_next/static|favicon.ico|sw.js).*)',
                headers: [
                    {
                        key: 'Cache-Control',
                        // 关键：不让浏览器和CDN缓存HTML
                        //value: 'no-store, no-cache, max-age=0, must-revalidate',
                        value: 'public, s-maxage=60, stale-while-revalidate=300',
                        // 或者使用更现代的策略：
                        // value: 's-maxage=1, stale-while-revalidate=59',
                    },
                ],
            },
        ];
    },
    compress: true,
    swcMinify: true,
    reactStrictMode: false,
    productionBrowserSourceMaps: false,
    transpilePackages: [
        'swiper',
        'ssr-window',
        'dom7',
        'axios-retry',
        'embla-carousel-react',
        'tailwind-merge',
        'clsx',
        'react-copy-to-clipboard',
        'react-infinite-scroll-component',
        'mitt'
    ],
    webpack(config, { isServer, dev }) {
        if (!isServer && !dev) {
            // 1. 修改 JS chunk 的文件名哈希长度
            // Next.js 默认的 chunkFilename 大概是 'static/chunks/[name]-[contenthash].js'
            // 我们需要把它改成 'static/chunks/[name]-[contenthash:8].js'
            if (config.output.chunkFilename.includes('[contenthash]')) {
                config.output.chunkFilename = 'static/chunks/[name]-[contenthash:8].js';
            }

            // 2. (可选) 修改主入口 JS 文件的文件名哈希长度
            // 这通常影响的是 pages/_app, pages/_document 等核心文件
            if (config.output.filename.includes('[contenthash]')) {
                config.output.filename = 'static/chunks/[name]-[contenthash:8].js';
            }
        }

        // 必须返回 config 对象
        return config;
    },
    /* async rewrites() {
        return process.env.NODE_ENV === 'development' ? [
            {
                source: '/dy-svr/:path*', // 需要代理的请求路径，例如 /api 路径下的请求
                destination: 'https://a.bdrcp.cn/dy-svr/:path*', // 代理到的目标地址
            },
            {
                source: '/json/:path*', // 需要代理的请求路径，例如 /api 路径下的请求
                destination: 'https://a.bdrcp.cn/:path*', // 代理到的目标地址
            },
            {
                source: '/si/:path*', // 需要代理的请求路径，例如 /interface 路径下的请求
                destination: 'https://a.bdrcp.cn/:path*', // 代理到的目标地址
            },
            {
                source: '/upload/:path*', // 需要代理的请求路径，例如 /upload 路径下的请求
                destination: 'https://a.bdrcp.cn/upload/:path*', // 代理到的目标地址
            },
        ] : []
    }, */
};

export default withBundleAnalyzer(nextConfig);